from Date import Date
from Recurrences import Recurrences



#
# Class for calendar events.
#
class CalEvent:

    def __init__(self):

        # (y,m,d) -> date
        self.__occur_table = {}

        # the recurrence rules of this event
        self.__recs = Recurrences()

        # start and end time
        self.__dtstart = Date(1970, 1, 1, 0, 0, 0)
        self.__dtend = Date(1970, 1, 1, 0, 0, 0)

        self.__summary = ""
        self.__location = ""
        self.__description = ""
        self.__categories = []

        self.__current_year = 0



    #
    # Computes the recurrences of this event for the given year.
    #
    def __compute_recurrences(self, year):

        self.__current_year = year
        self.__occur_table = {}
        recs = []
        
        if (self.__recs):
            # get rule-based recurrences
            recs = self.__recs.get_recurrences(self.get_start(), year)

            # get additional recurrences
            for date in self.__recs.get_dates():
                ryear, rmonth, rday = date.get_day()
                if (not (year - 1 <= ryear <= year)): continue
                recs.append(date.copy())
            #end for

        else:
            recs.append(self.get_start())

        recs = self.__bloat_recurrences(recs, year)
        for r in recs: self.__occur_table[r.get_day()] = r



    #
    # Inflates every recurrence to occupy as much time as the duration of this
    # events is.
    #
    def __bloat_recurrences(self, recs, year):

        duration = self.get_duration()
        bloated_recs = []

        for date in recs:
            bloated_recs.append(date.copy())

            dsecs = duration % 60
            duration /= 60
            dmins = duration % 60
            duration /= 60
            dhours = duration % 24
            duration /= 24
            ddays = duration

            end_date = date.copy()
            end_date.add_time(0, 0, ddays, dhours, dmins, dsecs)
            new_date = date.copy()
            new_date.add_time(0, 0, 1)

            # exclude the last day since the iCal specification says so
            loop_counter = 0
            while (new_date < end_date):
                y, m, d = new_date.get_day()
                nd = Date(y, m, d)
                bloated_recs.append(nd)
                new_date = new_date.copy()
                new_date.add_time(0, 0, 1)

                # work around a Python 2.2.3 bug on FreeBSD
                # (thanks, Tero Koskinen)
                loop_counter += 1
                if (loop_counter > 10000):
                    print "Your version of Python has a bug.\nCalendar " \
                          "events with recurrences will not be available."
                    break
            #end while
        #end for

        return bloated_recs




    #
    # Returns whether this event occurs on the given day.
    #
    def occurs_on(self, year, month, day):

        if (year != self.__current_year):
            self.__compute_recurrences(year)

        return self.__occur_table.has_key((year, month, day))



    #
    # Returns the occurrence (i.e. start time) of this event on the given day.
    # If day and month are omitted, a list of occurrences in the given year
    # will be returned.
    #
    def get_occurrence(self, year, month = 0, day = 0):

        if (year != self.__current_year):
            self.__compute_recurrences(year)

        if (not month):
            return self.__occur_table.values()
        else:
            return self.__occur_table.get((year, month, day), None)



    #
    # Returns the duration of this event in seconds.
    #
    def get_duration(self):

        duration = self.get_start().get_interval(self.get_end())
        return duration



    def set_recurrences(self, rrule): self.__recs = rrule
    def get_recurrences(self): return self.__recs

    def set_start(self, date): self.__dtstart = date
        
    def get_start(self): return self.__dtstart

    def set_end(self, date): self.__dtend = date
    def get_end(self): return self.__dtend

    def set_summary(self, text): self.__summary = text
    def get_summary(self): return self.__summary

    def set_location(self, text): self.__location = text
    def get_location(self): return self.__location

    def set_description(self, text): self.__description = text
    def get_description(self): return self.__description

    def set_categories(self, categs): self.__categories = categs
    def get_categories(self): return self.__categories


    #
    # Forces the recomputation of recurrences.
    #
    def invalidate(self): self.__current_year = 0


    def day_comparator(self, e1, e2):

        return cmp(t1, t2)



    def daytime_comparator(self, e1, e2):

        t1 = e1.get_start().get_daytime()
        t2 = e2.get_start().get_daytime()

        return cmp(t1, t2)
