/*
 * Argus-5.0 Client Software. Tools to read, analyze and manage Argus data.
 * Copyright (c) 2000-2024 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * $Id: //depot/gargoyle/clients/include/argus_event.h#5 $
 * $DateTime: 2014/05/14 00:30:13 $
 * $Change: 2825 $
 */
 
 
#ifndef Argus_event_h
#define Argus_event_h


#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
#include <argus_int.h>

#define ARGUS_ZLIB_COMPRESS	0x00000001
#define ARGUS_ZLIB_COMPRESS2	0x00000002

#define RA_MAX_TABLE_ARRAY	0x10000
#define RA_MAXTABLES		1
#define RA_MAXTABLES_MASK	0x00001

#define AIS_ETARGETS		4

#define AIS_DATABASE		0x01
#define AIS_SYSLOG		0x02
#define AIS_FILE		0x04
#define AIS_TERM		0x08


#define AIS_NTYPES		9
 
/* type codes */
#define AIS_EVENT		0x0001
#define AIS_CONDITION		0x0002
#define AIS_INTERNAL		0x0004
#define AIS_MONITOR		0x0008
#define AIS_REPORTED		0x0010
#define AIS_DERIVED		0x0020
#define AIS_OPERATIONS	0x0040
#define AIS_PERFORMANCE	0x0080
#define AIS_SECURITY		0x0100
 
#define AIS_NSEVERITY		9
 
/* severity codes */
#define AIS_EMERG		1       /* system is unusable */
#define AIS_ALERT		2       /* action must be taken immediately */
#define AIS_CRIT		3       /* critical conditions */
#define AIS_ERR		4       /* error conditions */
#define AIS_WARNING		5       /* warning conditions */
#define AIS_NOTICE		6       /* normal but significant condition */
#define AIS_INFO		7       /* informational */
#define AIS_DEBUG		8       /* debug-level messages */
 
#define AIS_NCAUSE            8
 
/* cause codes */
#define AIS_START		0
#define AIS_STATUS		1
#define AIS_STOP		2
#define AIS_TIMEOUT		3
#define AIS_SHUTDOWN		4
#define AIS_ERROR		5
#define AIS_OUTOFSPEC		6
#define AIS_INSPEC		7
 
#define AIS_NFACILITY		6

/* facilitity codes */
#define AIS_AIS		1
#define AIS_AISD		2
#define AIS_RADIUM		3
#define AIS_ISIS		4
#define AIS_RSVP		5
#define AIS_USER		6

 
#if defined(ARGUS_EVENT_PROC)
char *RaDataBase = "AIS";
unsigned int RaTableFlags = 0;

char *RaExistTableNames[RA_MAX_TABLE_ARRAY];

/*
  The AIS Database holds account information and entries for entire
  sets of ra data.  The schema includes an Events table which is
  designed to hold the local list of events that are generated by
  this ais.  Aspects of this table should be available to the
  composite aisd federation for correlation.

  AIS.Events schema.

+-----------+---------------+------+-----+---------------------+----------------+
| Field     | Type          | Null | Key | Default             | Extra          |
+-----------+---------------+------+-----+---------------------+----------------+
| id        | int(11)       |      | PRI | NULL                | auto_increment |
| aisid     | varchar(255)  |      |     |                     |                |
| uid       | int(11)       |      |     | 0                   |                |
| project   | int(11)       |      |     | 0                   |                |
| start     | timestamp(14) | YES  |     | NULL                |                |
| end       | timestamp(14) | YES  |     | 00000000000000      |                |
| facility  | int(11)       | YES  |     | NULL                |                |
| severity  | int(8)        | YES  |     | NULL                |                |
| access    | timestamp(14) | YES  |     | 00000000000000      |                |
| timestamp | datetime      |      |     | 0000-00-00 00:00:00 |                |
| hostname  | varchar(255)  | YES  |     | NULL                |                |
| sender    | varchar(255)  | YES  |     | NULL                |                |
| instance  | varchar(255)  | YES  |     | NULL                |                |
| version   | int(2)        |      |     | 0                   |                |
| message   | text          | YES  |     | NULL                |                |
| metadata  | text          | YES  |     | NULL                |                |
| status    | int(4)        | YES  |     | NULL                |                |
| record    | blob          | YES  |     | NULL                |                |
+-----------+---------------+------+-----+---------------------+----------------+


  The events schema is primarily designed to support syslog messages, which
  are a form of event that the ntias needs to support.

  The fields are:
     id       - the internal rdbms index number for the specific Entry.
     aisid    - the id for the ais that generated this event (url).
     uid      - user id that generated the event (ais specific).
     project  - project that generated the event (project id) (0 = ais).
     start    - Time when this event/condition started.
     end      - Time when this event/condition ended.
     access   - Time when this aisd received the event.

     version  - Event version number.
     type     - AIS event classification.
           Numerical  Class
             Code
              1       AIS Internal Component Event
              2       Monitored Network Entity Event
              3       Reported Network Entity State Transition
              4       Derived Network Entity State Transition

     cause    - reason event was generated.
           Numerical  Class
             Code
              1       Start    - initial report
              2       Status   - continuation record
              3       Stop     - condition close/termination indication
              4       Timeout  - timeout condition reached
              5       Shutdown - ais component shutdown
              6       Error    - ais error condition
              7       OutSpec  - value exceeded specification
              8       InSpe    - value within specificiation

     facility - event source classification.  this uses syslog.h for
                0-23 definitions in order to make importing syslog
                messages from routers/switches/end-systems straight
                forward.  AIS facility codes will be generated to
                provide greater utility.

     severity - event classification.  This number can be modified by the
                receiving aisd.  The complete original message is
                contained in the 'message' field, and so can be recovered.
           Numerical  Severity
             Code
              1       Emergency: system is unusable
              2       Alert: action must be taken immediately
              3       Critical: critical conditions
              4       Error: error conditions
              5       Warning: warning conditions
              6       Notice: normal but significant condition
              7       Informational: informational messages
              8       Debug: debug-level messages

    timestamp - The time reported by the generator of the event (syslog).
     hostname - Generally, this is the FQDN of the host/probe/system
                that generated the event.
       sender - This identifies the device or application that
                generated the event.
     instance - This identifies a specific instance of the sender.
                NULL value indicates an unidentified instance.

     message  - Complete event message received.
    metadata  - Complete event message received.
      status  - An internal ais status indicator (processed/tracked/imported/learned/deleted/orphaned).
*/

char *ArgusEventTableCreationString[RA_MAXTABLES * 2] = {
   "CREATE TABLE %s (id int not null auto_increment, aisid varchar(255) not null, uid int(11) not null, project int(11) not null, start datetime, end datetime, type int(8), cause int(8), facility int(11), severity int(8), access timestamp(14), timestamp datetime not null, hostname varchar(255), sender varchar(255), instance varchar(255), version int(2) not null, message text, metadata text, status int (4), record blob, primary key (id))",

   "CREATE TABLE %s (id int not null auto_increment, %s , aisid varchar(255) not null, uid int(11) not null, project int(11) not null, start datetime, end datetime, type int(8), cause int(8), facility int(11), severity int(8), access timestamp(14), timestamp datetime not null, hostname varchar(255), sender varchar(255), instance varchar(255), version int(2) not null, message text, metadata text, status int (4), record blob, primary key (id))",
};


struct ArgusTokenStruct AisEventTargets[AIS_ETARGETS] = {
   { AIS_DATABASE,   "sql" } ,
   { AIS_SYSLOG,     "syslog" } ,
   { AIS_FILE,       "file" } ,
   { AIS_TERM,       "term" } ,
};

struct ArgusTokenStruct RaSQLEventTypes[AIS_NTYPES] = {
   { AIS_EVENT,       "event" } ,
   { AIS_CONDITION,   "condition" } ,
   { AIS_INTERNAL,    "internal" } ,
   { AIS_MONITOR,     "monitor" } ,
   { AIS_REPORTED,    "reported" } ,
   { AIS_DERIVED,     "derived" } ,
   { AIS_OPERATIONS,  "ops" } ,
   { AIS_PERFORMANCE, "perf" } ,
   { AIS_SECURITY,    "sec" } ,
};

struct ArgusTokenStruct RaSQLEventCause[AIS_NCAUSE] = {
   { AIS_START,     "start" } ,
   { AIS_STATUS,    "status" } ,
   { AIS_STOP,      "stop" } ,
   { AIS_TIMEOUT,   "timeout" } ,
   { AIS_SHUTDOWN,  "shutdown" } ,
   { AIS_ERROR,     "error" } ,
   { AIS_OUTOFSPEC, "outofspec" } ,
   { AIS_INSPEC,    "inspec" } ,
};

struct ArgusTokenStruct RaSQLEventFacilities[AIS_NFACILITY] = {
   { AIS_AIS,       "ais" } ,
   { AIS_AISD,      "aisd" } ,
   { AIS_RADIUM,    "radium" } ,
   { AIS_RSVP,      "rsvp" } ,
   { AIS_ISIS,      "isis" } ,
   { AIS_USER,      "user" } ,
};

struct ArgusTokenStruct RaSQLEventSeverities[AIS_NSEVERITY] = {
   { 0,              "no" } ,
   { AIS_EMERG,    "emerg" } ,
   { AIS_ALERT,    "alert" } ,
   { AIS_CRIT,     "crit" } ,
   { AIS_ERR,      "err" } ,
   { AIS_WARNING,  "warn" } ,
   { AIS_NOTICE,   "notice" } ,
   { AIS_INFO,     "info" } ,
   { AIS_DEBUG,    "debug" } ,
};

#else

extern struct ArgusTokenStruct AisEventTargets[AIS_ETARGETS];
extern struct ArgusTokenStruct RaSQLEventTypes[AIS_NTYPES];
extern struct ArgusTokenStruct RaSQLEventCause[AIS_NCAUSE];
extern struct ArgusTokenStruct RaSQLEventFacilities[AIS_NFACILITY];
extern struct ArgusTokenStruct RaSQLEventSeverities[AIS_NSEVERITY];

#endif

struct ArgusEventObject {
   struct ArgusQueueHeader qhdr;
   int status, target;
   int type, cause, facility, severity;
   char *accounts, *message, *metadata;
};

struct ArgusEventObject *ArgusNewEvent(struct ArgusParserStruct *, int, int, int, int, int, char *, char *, char *);
void ArgusDeleteEvent(struct ArgusEventObject *);
int ArgusProcessEvent (struct ArgusParserStruct *, struct ArgusEventObject *, struct ArgusRecordStruct *);

#ifdef __cplusplus
}
#endif

#endif
